/*********************************************************************
 * Módulo 2. Curso de Experto en Desarrollo de Videojuegos
 * Autor: Carlos González Morcillo     Carlos.Gonzalez@uclm.es
 *
 * You can redistribute and/or modify this file under the terms of the
 * GNU General Public License ad published by the Free Software
 * Foundation, either version 3 of the License, or (at your option)
 * and later version. See <http://www.gnu.org/licenses/>.
 *
 * This file is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.  
 *********************************************************************/
#include "MyFrameListener.h"
#include "MyMotionState.h"

MyFrameListener::MyFrameListener(RenderWindow* win, 
				 Camera* cam, 
				 OverlayManager *om, 
				 SceneManager *sm) {
  OIS::ParamList param;
  size_t windowHandle;  std::ostringstream wHandleStr;

  _camera = cam;  _overlayManager = om; _sceneManager = sm;
  
  win->getCustomAttribute("WINDOW", &windowHandle);
  wHandleStr << windowHandle;
  param.insert(std::make_pair("WINDOW", wHandleStr.str()));
  
  _inputManager = OIS::InputManager::createInputSystem(param);
  _keyboard = static_cast<OIS::Keyboard*>
    (_inputManager->createInputObject(OIS::OISKeyboard, false));
  _mouse = static_cast<OIS::Mouse*>
    (_inputManager->createInputObject(OIS::OISMouse, false));
  _mouse->getMouseState().width = win->getWidth();
  _mouse->getMouseState().height = win->getHeight();

  _broadphase = new btDbvtBroadphase();
  _collisionConf = new btDefaultCollisionConfiguration();
  _dispatcher = new btCollisionDispatcher(_collisionConf);
  _solver = new btSequentialImpulseConstraintSolver;
  _world = new btDiscreteDynamicsWorld(_dispatcher,_broadphase,
				       _solver,_collisionConf);
  
  // Establecimiento propiedades del mundo
  _world->setGravity(btVector3(0,-10,0));

  // Creacion de los elementos iniciales del mundo
  CreateInitialWorld();
}

MyFrameListener::~MyFrameListener() {
  _inputManager->destroyInputObject(_keyboard);
  _inputManager->destroyInputObject(_mouse);
  OIS::InputManager::destroyInputSystem(_inputManager);

  _world->removeRigidBody(_fallRigidBody);
  delete _fallRigidBody->getMotionState();  delete _fallRigidBody;
  
  _world->removeRigidBody(_groundRigidBody);
  delete _groundRigidBody->getMotionState();  delete _groundRigidBody;
  
  delete _fallShape;      delete _groundShape;
  
  delete _world;  delete _solver;
  delete _collisionConf;
  delete _dispatcher;     delete _broadphase;
}

void MyFrameListener::CreateInitialWorld() {
  // Creacion de la entidad y del SceneNode ------------------------
  Plane plane1(Vector3(0,1,0), 0);    // Normal y distancia
  MeshManager::getSingleton().createPlane("p1",
	ResourceGroupManager::DEFAULT_RESOURCE_GROUP_NAME, plane1,
	200, 200, 1, 1, true, 1, 20, 20, Vector3::UNIT_Z);
  SceneNode* node = _sceneManager->createSceneNode("ground");
  Entity* groundEnt = _sceneManager->createEntity("planeEnt", "p1");
  groundEnt->setMaterialName("Ground");
  node->attachObject(groundEnt);
  _sceneManager->getRootSceneNode()->addChild(node);

  // Creamos las formas de colision --------------------------------- 
  _groundShape = new btStaticPlaneShape(btVector3(0,1,0),1);
  _fallShape = new btSphereShape(1);
  
  // Creamos el plano -----------------------------------------------
  MyMotionState* groundMotionState = new MyMotionState(
    btTransform(btQuaternion(0,0,0,1),btVector3(0,-1,0)), node);
  btRigidBody::btRigidBodyConstructionInfo
    groundRigidBodyCI(0,groundMotionState,_groundShape,btVector3(0,0,0));
  _groundRigidBody = new btRigidBody(groundRigidBodyCI);
  _world->addRigidBody(_groundRigidBody);
  
  // Creamos la esfera ----------------------------------------------
  Entity *entity2 = _sceneManager->createEntity("ball", "ball.mesh");
  SceneNode *node2 = _sceneManager->getRootSceneNode()->createChildSceneNode();
  node2->attachObject(entity2);
  MyMotionState* fallMotionState = new MyMotionState(
     btTransform(btQuaternion(0,0,0,1),btVector3(0,50,0)), node2);
  btScalar mass = 1;
  btVector3 fallInertia(0,0,0);
  _fallShape->calculateLocalInertia(mass,fallInertia);
  btRigidBody::btRigidBodyConstructionInfo fallRigidBodyCI(
     mass,fallMotionState,_fallShape,fallInertia);
  _fallRigidBody = new btRigidBody(fallRigidBodyCI);
  _world->addRigidBody(_fallRigidBody);
}

bool MyFrameListener::frameStarted(const Ogre::FrameEvent& evt) {
  Ogre::Vector3 vt(0,0,0);     Ogre::Real tSpeed = 20.0;  
  Ogre::Real deltaT = evt.timeSinceLastFrame;
  int fps = 1.0 / deltaT;
  bool mbleft, mbmiddle, mbright; // Botones del raton pulsados

  _world->stepSimulation(deltaT, 1); // Actualizar simulacion Bullet

  _keyboard->capture();
  if (_keyboard->isKeyDown(OIS::KC_ESCAPE)) return false;


  btVector3 impulse(0,0,0);
  if (_keyboard->isKeyDown(OIS::KC_I)) {_fallRigidBody->activate(true); impulse = btVector3(0,0,-.1);}
  if (_keyboard->isKeyDown(OIS::KC_J)) { _fallRigidBody->activate(true); impulse = btVector3(-.1,0,0);}
  if (_keyboard->isKeyDown(OIS::KC_K)){ _fallRigidBody->activate(true); impulse = btVector3(0,0,.1);}
  if (_keyboard->isKeyDown(OIS::KC_L)) { _fallRigidBody->activate(true); impulse = btVector3(.1,0,0);}
  _fallRigidBody->applyCentralImpulse(impulse);

  int posx = _mouse->getMouseState().X.abs;   // Posicion del puntero
  int posy = _mouse->getMouseState().Y.abs;   //  en pixeles.

  _camera->moveRelative(vt * deltaT * tSpeed);
  if (_camera->getPosition().length() < 10.0) {
    _camera->moveRelative(-vt * deltaT * tSpeed);
  }

  _mouse->capture();

 // Si usamos la rueda, desplazamos en Z la camara ------------------
  vt+= Ogre::Vector3(0,0,-10)*deltaT * _mouse->getMouseState().Z.rel;   
  _camera->moveRelative(vt * deltaT * tSpeed);

  // Botones del raton pulsados? -------------------------------------
  mbleft = _mouse->getMouseState().buttonDown(OIS::MB_Left);
  mbmiddle = _mouse->getMouseState().buttonDown(OIS::MB_Middle);
  mbright = _mouse->getMouseState().buttonDown(OIS::MB_Right);

  if (mbmiddle) { // Con boton medio pulsado, rotamos camara ---------
    float rotx = _mouse->getMouseState().X.rel * deltaT * -1;
    float roty = _mouse->getMouseState().Y.rel * deltaT * -1;
    _camera->yaw(Ogre::Radian(rotx));
    _camera->pitch(Ogre::Radian(roty));
  }

  Ogre::OverlayElement *oe;
  oe = _overlayManager->getOverlayElement("cursor");
  oe->setLeft(posx);  oe->setTop(posy);

  oe = _overlayManager->getOverlayElement("fpsInfo");
  oe->setCaption(Ogre::StringConverter::toString(fps));

  return true;
}

bool MyFrameListener::frameEnded(const Ogre::FrameEvent& evt) {
  Real deltaT = evt.timeSinceLastFrame;
  _world->stepSimulation(deltaT, 1); // Actualizar simulacion Bullet
  return true;
}
